function NavNode()
{
	// 13 mutable members (will change as student interacts
	// with the courseware).
	this.skip = null;
	this.block = null;
	this.enabled = null;
	this.visible = null;
	
	this.completed = null;
	this.attempted = null;
	this.objective = null;
	this.parentStack = null;
	this.selectedChildren = null;
	this.position = null;
	
	// Control Modes
	this.cFlow = null;
	this.cChoice = null;
	this.cForwardOnly = null;
	
	this.skipDefault = null;
	this.blockDefault = null;
	this.enabledDefault = null;
	this.visibleDefault = null;
	this.choiceDefault = null;
	this.flowDefault = null;
	this.forwardOnlyDefault = null;
	
	// immutable members (will not change as student interacts
	// with the courseware).
	this.id = null;
	
	// 0-based. This num value is used as a way to determine
	// the relative order of nodes in the forest, therefore a
	// node's num value will change as selection occurs on
	// its ancestor nodes.
	this.num = null;
	
	this.url = null;
	this.tag = null;
	this.title = null;
	this.weight = null; // range = [0,1]
	this.children = null;
	this.objectiveMaps = null;
	this.navStateRules = null;
	this.postConditionRules = null;
	this.transparent = null;
	// Selection Controls
	this.sRandomize = null;
	this.sSelect = null;
	this.sSelectEvent = null;
	
	this.events = null;
	this.navEngineRef = null;
}
NavNode.prototype.loadFromXMLNode = NavNodeLoadFromXMLNode;
NavNode.prototype.setDefaultValues = NavNodeSetDefaultValues;
NavNode.prototype.toString = NavNodeToString;
NavNode.prototype.init = NavNodeInit;
NavNode.prototype.reset = NavNodeReset;
NavNode.prototype.resolveReferences = NavNodeResolveReferences;
NavNode.prototype.getSatisfied = NavNodeGetSatisfied;
NavNode.prototype.setSatisfied = NavNodeSetSatisfied;
NavNode.prototype.getScore = NavNodeGetScore;
NavNode.prototype.setScore = NavNodeSetScore;
NavNode.prototype.getAttempted = NavNodeGetAttempted;
NavNode.prototype.setAttempted = NavNodeSetAttempted;
NavNode.prototype.getCompleted = NavNodeGetCompleted;
NavNode.prototype.setCompleted = NavNodeSetCompleted;
NavNode.prototype.getSkip = NavNodeGetSkip;
NavNode.prototype.setSkip = NavNodeSetSkip;
NavNode.prototype.getChoice = NavNodeGetChoice;
NavNode.prototype.setChoice = NavNodeSetChoice;
NavNode.prototype.getFlow = NavNodeGetFlow;
NavNode.prototype.setFlow = NavNodeSetFlow;
NavNode.prototype.getForwardOnly = NavNodeGetForwardOnly;
NavNode.prototype.setForwardOnly = NavNodeSetForwardOnly;
NavNode.prototype.getBlock = NavNodeGetBlock;
NavNode.prototype.setBlock = NavNodeSetBlock;
NavNode.prototype.getEnabled = NavNodeGetEnabled;
NavNode.prototype.setEnabled = NavNodeSetEnabled;
NavNode.prototype.getVisible = NavNodeGetVisible;
NavNode.prototype.setVisible = NavNodeSetVisible;
NavNode.prototype.registerEvent = RegisterEvent;
NavNode.prototype.calculateScore = NavNodeCalculateScore;
NavNode.prototype.getPostAction = NavNodeGetPostAction;
NavNode.prototype.callbackChildAttempted = NavNodeCallbackChildAttempted;
NavNode.prototype.callbackScore = function (ref){this.events.raiseEvent("score", this);};
NavNode.prototype.callbackSatisfied = function (ref){this.events.raiseEvent("satisfied", this);};
NavNode.prototype.getNextSibling = NavNodeGetNextSibling;
NavNode.prototype.getPreviousSibling = NavNodeGetPreviousSibling;
NavNode.prototype.eventMonitor = NavNodeEventMonitor;

NavNode.prototype.getId = NavNodeGetId;
NavNode.prototype.getTitle = NavNodeGetTitle;
NavNode.prototype.getTag = NavNodeGetTag;
NavNode.prototype.getUrl = NavNodeGetUrl;
NavNode.prototype.containsNode = NavNodeContainsNode;

function NavNodeContainsNode(id)
{
	if (this.id == id)
	{
		return true;
	}
	else
	{
		for (var i=0;i<this.children.length;i++)
		{
			if (this.children[i].containsNode(id))
			{
				return true;
			}
		}
	}
	return false;
}

function NavNodeGetId()
{
	return this.id;
}

function NavNodeGetTitle()
{
	return this.title;
}

function NavNodeGetTag()
{
	return this.tag;
}

function NavNodeGetUrl()
{
	return this.url;
}

function NavNodeGetPreviousSibling(bolUseSelected)
{
	if (this.parentStack[0] == null) return null;
	var childArray = this.parentStack[0].children;
	if (bolUseSelected && this.parentStack[0].selectedChildren.length > 0)
	{
		childArray = this.parentStack[0].selectedChildren;
	}

	for (var i=1;i<childArray.length;i++)
	{
		if (childArray[i] == this)
		{
			return childArray[i-1];
		}
	}
	return null;
}

function NavNodeGetNextSibling(bolUseSelected)
{
	if (this.parentStack[0] == null) return null;
	var childArray = this.parentStack[0].children;
	if (bolUseSelected && this.parentStack[0].selectedChildren.length > 0)
	{
		childArray = this.parentStack[0].selectedChildren;
	}

	for (var i=0;i<childArray.length-1;i++)
	{
		if (childArray[i] == this)
		{
			return childArray[i+1];
		}
	}
	return null;
}

//****************************************************************************
//
// NavNodeCallbackChildAttempted()
// Callback function that is called in response to a child node's "attempted"
// event (the "attempted" event is fired whenever a node's attempted property
// changes). The purpose of the function is to catch these "attempted" events,
// and then change the node's own attempted property if needed.
//
// Input Params: flag - Boolean - specifies what the child's attempted
//                                property was changed to.
//
// Output Params: None
//
// Returns: None
//
// Calls/References: None
//
//****************************************************************************
function NavNodeCallbackChildAttempted(ref)
{
	var flag = ref.getAttempted();
	// If the flag is true, then set the node's own
	// attempted property to true as well.
	if (flag == true)
	{
		this.setAttempted(true);
	}
	// If the flag is false, and the node's own
	// attempted property is true, then it will
	// need to be determined whether or not the
	// node's own attempted property should be
	// changed to false as well.
	else if (this.getAttempted())
	{
		var childArray = this.children;
		// Loop through all child node's, and
		// return immediately if any are found
		// to have an attempted property of true.
		for (var i=0;i<childArray.length;i++)
		{
			if (childArray[i].getAttempted())
			{
				return;
			}
		}
		
		// If processing has reached this line,
		// then all children's attempted properties
		// are false, and the node's own attempted
		// property should be set to false as well.
		this.setAttempted(false);
	}
}

//****************************************************************************
//
// NavNodeGetPostAction()
// Function that is used to determine what post actions are required by the
// node, as determined by the evaluation of the node's post condition rules
// at the time that the call is made.
//
// Input Params: direction - ENUM - specifies the navigational direction.
//                                  (NAV_DIRECTION_FORWARD, or
//                                  NAV_DIRECTION_REVERSE)
//
// Output Params: None
//
// Returns: Object -  contains value and parameters properties.  The value 
//                    property is an integer with at most 2 bits set: the 
//                    NAV_POST_ACTION_RESET bit, and any one of the other
//                    NAV_POST_ACTION bits.  It is also possible that no bits
//                    would be set, which denotes that no specific post actions
//                    are required.  The parameters property is an array containing
//                    any parameters required for the specific action.
//
// Calls/References: None
//
//****************************************************************************
function NavNodeGetPostAction(direction)
{
	var retVal = new Object();
	retVal.value = 0;
	retVal.parameters = new Array();
	
	// First loop through all post condition rules, looking for
	// any that have NAV_POST_ACTION_RESET as their action.  If
	// one is found that evaluates to true, then set the 
	// NAV_POST_ACTION_RESET bit in the retVal variable.
	for (var i=0;i<this.postConditionRules.length;i++)
	{
		if ((this.postConditionRules[i].direction & direction) > 0 && this.postConditionRules[i].action == NAV_POST_ACTION_RESET)
		{
			if (this.postConditionRules[i].evaluate() == true)
			{
				retVal.value = retVal.value | NAV_POST_ACTION_RESET;
				break;
			}
		}
	}
	
	// Next loop through all post condition rules, looking for
	// the first non-NAV_POST_ACTION_RESET one that evaluate to
	// true (only one non-NAV_POST_ACTION_RESET rule can be used). 
	for (var i=0;i<this.postConditionRules.length;i++)
	{
		if ((this.postConditionRules[i].direction & direction) > 0 && this.postConditionRules[i].action != NAV_POST_ACTION_RESET)
		{
			if (this.postConditionRules[i].evaluate() == true)
			{
				retVal.value = retVal.value | this.postConditionRules[i].action;
				retVal.parameters = this.postConditionRules[i].params.slice(0,this.postConditionRules[i].params.length);
				break;
			}
		}
	}
	
	return retVal;
}

//****************************************************************************
//
// NavNodeCalculateScore()
// Function that is called to determine and then set a node's score.  This 
// function is used as a callback in response to any of the node's children
// firing its "score" event.
//
// Input Params: None
//
// Output Params: None
//
// Returns: None
//
// Calls/References: None
//
//****************************************************************************
function NavNodeCalculateScore(ref)
{
	// Score is calculated by taking the sum of the products of all the 
	// children's scores and weights, and dividing by the sum of all
	// the children's weights:
	// 
	//    (score(1)*weight(1)) + (score(2)*weight(2)) + ... + (score(n)*weight(n))
	// _______________________________________________________________________________
	//
	//                   weight(1) + weight(2) + ... + weight(n)
	//
	// where n is the number of child nodes considered in the calculation.

	// Accumulator for all of the (score(i)*weight(i)) values.
	var scores = 0;
	// Accumulator for all of the weight(i) values.
	var weights = 0;
	
	// First determine the set of children to use.
	var childArray = null;	
	if (this.selectedChildren.length > 0)
	{
		childArray = this.selectedChildren;
	}
	else
	{
		childArray = this.children;
	}
	
	// Perform the calculation.
	for (var i=0;i<childArray.length;i++)
	{
		scores += childArray[i].weight * childArray[i].getScore();
		weights += childArray[i].weight;
	}
	this.setScore(scores/weights);
}

function NavNodeGetSkip()
{
	return this.skip;
}

function NavNodeSetSkip(flag)
{
	if (this.skip != flag)
	{
		this.skip = flag;
		this.events.raiseEvent("skip",this);
	}
}

function NavNodeGetChoice()
{
	return this.cChoice;
}

function NavNodeSetFlow(flag)
{
	if (this.cFlow != flag)
	{
		this.cFlow = flag;
		this.events.raiseEvent("flow",this);
	}
}

function NavNodeGetFlow()
{
	return this.cFlow;
}

function NavNodeSetForwardOnly(flag)
{
	if (this.cForwardOnly != flag)
	{
		this.cForwardOnly = flag;
		this.events.raiseEvent("forwardOnly",this);
	}
}

function NavNodeGetForwardOnly()
{
	return this.cForwardOnly;
}

function NavNodeSetChoice(flag)
{
	if (this.cChoice != flag)
	{
		this.cChoice = flag;
		this.events.raiseEvent("choice",this);
	}
}

function NavNodeGetBlock()
{
	return this.block;
}

function NavNodeSetBlock(flag)
{
	if (this.block != flag)
	{
		this.block = flag;
		this.events.raiseEvent("block",this);
	}
}

function NavNodeGetEnabled()
{
	return this.enabled;
}

function NavNodeSetEnabled(flag)
{
	if (this.enabled != flag)
	{
		this.enabled = flag;
		this.events.raiseEvent("enabled",this);
	}
}

function NavNodeGetVisible()
{
	return this.visible;
}

function NavNodeSetVisible(flag)
{
	if (this.visible != flag)
	{
		this.visible = flag;
		this.events.raiseEvent("visible",this);
	}
}

function NavNodeGetAttempted()
{
	return this.attempted;
}

function NavNodeSetAttempted(flag)
{
	if (this.attempted != flag)
	{
		this.attempted = flag;
		this.events.raiseEvent("attempted",this);
	}
}

function NavNodeGetCompleted()
{
	return this.completed;
}

function NavNodeSetCompleted(flag)
{
	if (this.completed != flag)
	{
		this.completed = flag;
		this.events.raiseEvent("completed",this);
	}
}

function NavNodeGetScore()
{
	for (var i=0;i<this.objectiveMaps.length;i++)
	{
		if (this.objectiveMaps[i].read == true)
		{
			return this.objectiveMaps[i].objRef.getScore();
		}
	}
	return this.objective.getScore();
}

function NavNodeSetScore(score)
{
	score = floatParse(score,SCORE_PRECISION);
	if (this.getScore() != score)
	{
		for (var i=0;i<this.objectiveMaps.length;i++)
		{
			if (this.objectiveMaps[i].write == true)
			{
				this.objectiveMaps[i].objRef.setScore(score);
			}
		}
		this.objective.setScore(score);
	}
}

function NavNodeGetSatisfied()
{
	for (var i=0;i<this.objectiveMaps.length;i++)
	{
		if (this.objectiveMaps[i].read == true)
		{
			return this.objectiveMaps[i].objRef.getSatisfied();
		}
	}
	return this.objective.getSatisfied();
}

function NavNodeSetSatisfied(flag)
{
	for (var i=0;i<this.objectiveMaps.length;i++)
	{
		if (this.objectiveMaps[i].write == true)
		{
			this.objectiveMaps[i].objRef.setSatisfied(flag);
		}
	}
	this.objective.setSatisfied(flag);
}

function NavNodeResolveReferences(navRef)
{
	var eventRegistered = false;
	for (var i=0;i<this.objectiveMaps.length;i++)
	{
		this.objectiveMaps[i].objRef = navRef.getObjectiveById(this.objectiveMaps[i].objRef);
		if (this.objectiveMaps[i].read && !eventRegistered)
		{
			this.objectiveMaps[i].objRef.registerEvent("score", "callbackScore", this,null,true);
			this.objectiveMaps[i].objRef.registerEvent("satisfied", "callbackSatisfied", this,null,true);
			eventRegistered = true;
		}
	}
	if (!eventRegistered)
	{
		this.objective.registerEvent("score", "callbackScore", this,null,true);
		this.objective.registerEvent("satisfied", "callbackSatisfied", this,null,true);
	}
	
	for (var i=0;i<this.navStateRules.length;i++)
	{
		this.navStateRules[i].target = this;
		this.navStateRules[i].resolveReferences(navRef);
	}
	
	for (var i=0;i<this.postConditionRules.length;i++)
	{
		this.postConditionRules[i].resolveReferences(navRef);
	}
	
	for (var i=0;i<this.children.length;i++)
	{
		this.children[i].resolveReferences(navRef);
		this.children[i].registerEvent("attempted","callbackChildAttempted",this,null,true);
	}
}

// Initialize the node, and all sub-nodes.
function NavNodeInit()
{
	if (this.sSelect > 0 && this.sSelectEvent == NAV_EVENT_INIT)
	{
		var bag = new Bag();
		this.selectedChildren = new Array();
		for (var i=0;i<this.children.length;i++)
		{
			bag.insert(this.children[i]);
		}
		for (var i=0;i<this.sSelect;i++)
		{
			this.selectedChildren[this.selectedChildren.length] = bag.remove();
		}
		this.events.raiseEvent("reorder",this);
	}
	
	for (var i=0;i<this.children.length;i++)
	{
		this.children[i].init();
	}
}

// Reset the node and all sub-nodes.
function NavNodeReset()
{
	if (this.sSelect > 0 && this.sSelectEvent == NAV_EVENT_RESET)
	{
		var bag = new Bag();
		this.selectedChildren = new Array();
		for (var i=0;i<this.children.length;i++)
		{
			bag.insert(this.children[i]);
		}
		for (var i=0;i<this.sSelect;i++)
		{
			this.selectedChildren[this.selectedChildren.length] = bag.remove();
		}
		this.events.raiseEvent("reorder",this);
	}
	else if (this.sRandomize)
	{
		if (this.selectedChildren.length > 0)
		{
			var bag = new Bag();
			for (var i=0;i<this.selectedChildren.length;i++)
			{
				bag.insert(this.selectedChildren[i]);
			}
			this.selectedChildren = new Array();
			while (bag.getLength() > 0)
			{
				this.selectedChildren[this.selectedChildren.length] = bag.remove();
			}
		}
		else
		{
			var bag = new Bag();
			for (var i=0;i<this.children.length;i++)
			{
				bag.insert(this.children[i]);
			}
			while (bag.getLength() > 0)
			{
				this.selectedChildren[this.selectedChildren.length] = bag.remove();
			}
		}
		this.events.raiseEvent("reorder",this);
	}

	this.setSkip(this.skipDefault);
	this.setChoice(this.choiceDefault);
	this.setFlow(this.flowDefault);
	this.setForwardOnly(this.forwardOnlyDefault);
	this.setBlock(this.blockDefault);
	this.setEnabled(this.enabledDefault);
	this.setVisible(this.visibleDefault);
	this.setCompleted(false);
	this.setAttempted(false);
	this.setSatisfied(false);
	this.setScore(0);
	
	for (var i=0;i<this.children.length;i++)
	{
		this.children[i].reset();
	}
	
}

function NavNodeToString(num)
{
	var strTab = "";
	var str = "";
	var strChildren = "";
	var strSelectedChildren = "";
	var strParentStack = "";
	var strObjMaps = "";
	var strNavStateRules = "";
	var strPostRules = "";

	for (var i=0;i<this.postConditionRules.length;i++)
	{
		strPostRules += this.postConditionRules[i].toString() + " | ";
	}
	if (strPostRules.length > 3) strPostRules = strPostRules.slice(0,strPostRules.length-3);
	
	for (var i=0;i<this.navStateRules.length;i++)
	{
		strNavStateRules += this.navStateRules[i].toString() + " | ";
	}
	if (strNavStateRules.length > 3) strNavStateRules = strNavStateRules.slice(0,strNavStateRules.length-3);
	
	for (var i=0;i<this.objectiveMaps.length;i++)
	{
		strObjMaps += "(objRef='" + this.objectiveMaps[i].objRef.id + "', " + ((this.objectiveMaps[i].read)?("read"):("no read")) + ", " + ((this.objectiveMaps[i].write)?("write"):("no write")) + ") | ";
	}
	if (strObjMaps.length > 3) strObjMaps = strObjMaps.slice(0,strObjMaps.length-3);
	
	for (var i=0;i<this.parentStack.length;i++)
	{
		strParentStack += this.parentStack[i].id + " | ";
	}
	if (strParentStack.length > 3) strParentStack = strParentStack.slice(0,strParentStack.length-3);
	
	for (var i=0;i<this.children.length;i++)
	{
		strChildren += this.children[i].id + " | ";
	}
	if (strChildren.length > 3) strChildren = strChildren.slice(0,strChildren.length-3);
	
	for (var i=0;i<this.selectedChildren.length;i++)
	{
		strSelectedChildren += this.selectedChildren[i].id + " | ";
	}
	if (strSelectedChildren.length > 3) strSelectedChildren = strSelectedChildren.slice(0,strSelectedChildren.length-3);
	
	if (num == null) num = 0;
	for (var i=0;i<num;i++) strTab += "\t";
	
	str = strTab + "[NavNode: " + "id=\"" + this.id + "\", " + "num=\"" + this.num + "\", " + "title=\"" + this.title + "\", " + "tag=\"" + this.tag + "\", " + "url=\"" + this.url + "\", " + "objective=\"(satisfied='" + this.objective.satisfied + "' score='" + this.objective.score +"')\", " + "objectiveMaps=\"" + strObjMaps + "\", " + "navStateRules=\"" + strNavStateRules + "\", " + "postConditionRules=\"" + strPostRules + "\", " + "weight=\"" + this.weight + "\", " + "flags=\"" + ((this.skip)?("skip | "):("don't skip | ")) + ((this.block)?("blocked | "):("not blocked | ")) + ((this.enabled)?("enabled | "):("disabled | ")) + ((this.visible)?("visible | "):("hidden | ")) + ((this.attempted)?("attempted | "):("not attempted | ")) + ((this.completed)?("completed"):("not completed")) + "\"" + ", " + "controlModes=\"" + ((this.cFlow)?("flow | "):("no flow | ")) + ((this.cChoice)?("choice | "):("no choice | ")) + ((this.cForwardOnly)?("forward only"):("forward/backward")) + "\"" + ", selectionControls=\"" + ((this.sRandomize)?("randomize on reset"):("no randomization")) + " | "  + ((this.sSelect>0)?("select " + this.sSelect + " of " + this.children.length + " on " + ((this.sSelectEvent==NAV_EVENT_INIT)?("init"):("")) + ((this.sSelectEvent==NAV_EVENT_RESET)?("reset"):(""))):("select all " + this.children.length)) + "\"" + ", parentStack=\"" + strParentStack + "\"" + ", children=\"" +  strChildren + "\"" + ", selectedChildren=\"" +  strSelectedChildren + "\"" + "]";
	
	return str;
}

function NavNodeSetDefaultValues()
{
	this.skip = false;
	this.block = false;
	this.enabled = true;
	this.visible = true;
	this.skipDefault = false;
	this.choiceDefault = true;
	this.flowDefault = true;
	this.forwardOnlyDefault = false;
	this.blockDefault = false;
	this.enabledDefault = true;
	this.visibleDefault = true;
	this.completed = false;
	this.attempted = false;
	this.transparent = false;
	
	this.id = "";
	this.num = -1;
	this.url = "";
	this.tag = "";
	this.title = "";
	this.objective = new NavObjective();
	this.weight = 1;
	
	this.position = new Array();	
	this.parentStack = new Array();
	this.children = new Array();
	this.selectedChildren = new Array();
	this.objectiveMaps = new Array();
	this.navStateRules = new Array();
	this.postConditionRules = new Array();	
	this.cFlow = true;
	this.cChoice = true;
	this.cForwardOnly = false;
		
	this.sRandomize = false;
	this.sSelect = 0;
	this.sSelectEvent = NAV_EVENT_INIT;

	this.events = new EventsObject(DEFAULT_NODE_EVENTS);
	this.events.owner = this;
	
	this.navEngineRef = null;
}

function NavNodeLoadFromXMLNode(xmlNode, navEngineRef)
{
	this.setDefaultValues();
	this.navEngineRef = navEngineRef;
	
	this.id = xmlNode.getAttribute("id");
	
	var url = xmlNode.getAttribute("url");
	if (url != null)
	{
		this.url = url;
	}
	
//	if ((xmlNode.getAttribute("transparent")+"").toLowerCase() == "true")
	if ((xmlNode.getAttribute("transparent")+"") == "true")
	{
		this.transparent = true;
	}
	
//	if ((xmlNode.getAttribute("skip")+"").toLowerCase() == "true")
	if ((xmlNode.getAttribute("skip")+"") == "true")
	{
		this.skip = true;
		this.skipDefault = true;
	}
	
//	if ((xmlNode.getAttribute("block")+"").toLowerCase() == "true")
	if ((xmlNode.getAttribute("block")+"") == "true")
	{
		this.block = true;
		this.blockDefault = true;
	}
	
//	if ((xmlNode.getAttribute("enabled")+"").toLowerCase() == "false")
	if ((xmlNode.getAttribute("enabled")+"") == "false")
	{
		this.enabled = false;
		this.enabledDefault = false;
	}
	
//	if ((xmlNode.getAttribute("visible")+"").toLowerCase() == "false")
	if ((xmlNode.getAttribute("visible")+"") == "false")
	{
		this.visible = false;
		this.visibleDefault = false;
	}
	
	var tag = xmlNode.getAttribute("tag");
	if (tag != null)
	{
		this.tag = tag;
	}

	var title = xmlNode.getAttribute("title");
	if (title != null)
	{
		this.title = title;
	}

	var weight = xmlNode.getAttribute("weight");
	if (weight != null)
	{
		this.weight = parseFloat(weight);
	}

		
//	if ((xmlNode.getAttribute("flow")+"").toLowerCase() == "false")
	if ((xmlNode.getAttribute("flow")+"") == "false")
	{
		this.cFlow = false;
		this.flowDefault = false;
	}
	
//	if ((xmlNode.getAttribute("choice")+"").toLowerCase() == "false")
	if ((xmlNode.getAttribute("choice")+"") == "false")
	{
		this.cChoice = false;
		this.choiceDefault = false;
	}
	
//	if ((xmlNode.getAttribute("forwardOnly")+"").toLowerCase() == "true")
	if ((xmlNode.getAttribute("forwardOnly")+"") == "true")
	{
		this.cForwardOnly = true;
		this.forwardOnlyDefault = true;
	}

	var arrMap = new Array();
	var arrNavStateRules = new Array();
	var arrPostStateRules = new Array();
	var arrSelectionControls = new Array();
	var arrChildren = new Array();
	if (xmlNode.children)
	{
		for (var i=0; i<xmlNode.children.length; i++)
		{
			if (xmlNode.children[i].nodeType=='ELEMENT')
			{
				switch (xmlNode.children[i].tagName)
				{
					case "objMap":arrMap[arrMap.length] = xmlNode.children[i];break;
					case "navStateRules":arrNavStateRules[arrNavStateRules.length] = xmlNode.children[i];break;
					case "postConditionRules":arrPostStateRules[arrPostStateRules.length] = xmlNode.children[i];break;
					case "selectionControls":arrSelectionControls[arrSelectionControls.length] = xmlNode.children[i];break;
					case "children":arrChildren[arrChildren.length] = xmlNode.children[i];break;
				}
			}
		}
	}

	if (arrMap.length > 0)
	{
		for (var i=0;i<arrMap[0].children.length;i++)
		{
			if (arrMap[0].children[i].nodeType == "ELEMENT")
			{
				var obj = new NavObjectiveMap();
				obj.loadFromXMLNode(arrMap[0].children[i]);
				this.objectiveMaps[this.objectiveMaps.length] = obj;
			}
		}
	}

	if (arrNavStateRules.length > 0)
	{
		for (var i=0;i<arrNavStateRules[0].children.length;i++)
		{
			if (arrNavStateRules[0].children[i].nodeType == "ELEMENT")
			{
				var navStateRule = new NavStateRule();
				navStateRule.loadFromXMLNode(arrNavStateRules[0].children[i]);
				this.navStateRules[this.navStateRules.length] = navStateRule;
			}
		}
	}

	if (arrPostStateRules.length > 0)
	{
		for (var i=0;i<arrPostStateRules[0].children.length;i++)
		{
			if (arrPostStateRules[0].children[i].nodeType == "ELEMENT")
			{
				var postStateRule = new NavPostRule();
				postStateRule.loadFromXMLNode(arrPostStateRules[0].children[i]);
				this.postConditionRules[this.postConditionRules.length] = postStateRule;
			}
		}
	}

	if (arrSelectionControls.length > 0)
	{
		for (var i=0;i<arrSelectionControls[0].children.length;i++)
		{
			if (arrSelectionControls[0].children[i].nodeType == "ELEMENT")
			{
				if (arrSelectionControls[0].children[i].tagName.toLowerCase() == "select")
				{
					var num = parseInt(arrSelectionControls[0].children[i].getAttribute("num"));
					var evt = arrSelectionControls[0].children[i].getAttribute("event");
					switch (evt.toLowerCase())
					{
						case "init":evt=NAV_EVENT_INIT;break;
						case "reset":evt=NAV_EVENT_RESET;break;
					}
					this.sSelect = num;
					this.sSelectEvent = evt;
				}
				else if (arrSelectionControls[0].children[i].tagName.toLowerCase() == "randomizeonreset")
				{
					this.sRandomize = true;
				}
			}
		}
	}	

	if (arrChildren.length > 0)
	{
		for (var i=0;i<arrChildren[0].children.length;i++)
		{
			if (arrChildren[0].children[i].nodeType == "ELEMENT")
			{
				var navNode = new NavNode();
				navNode.loadFromXMLNode(arrChildren[0].children[i], this.navEngineRef);
				navNode.parentStack[0] = this;
				navNode.registerEvent("score","calculateScore",this,null,true);
				this.children[this.children.length] = navNode;
			}
		}
	}
}

function NavNodeEventMonitor(strEvent, value1, value2)
{
	if (this.navEngineRef)
	{
		switch (strEvent.toLowerCase())
		{
			case "reorder":if (!this.registeredReorder) {this.registeredReorder = true;this.events.registerEvent("reorder","callbackReorder",this.navEngineRef,null,true);};break;
			case "attempted":if (!this.registeredAttempted) {this.registeredAttempted = true;this.events.registerEvent("attempted","callbackNodeAttempted",this.navEngineRef,null,true);};break;
			case "completed":if (!this.registeredCompleted) {this.registeredCompleted = true;this.events.registerEvent("completed","callbackNodeCompleted",this.navEngineRef,null,true);};break;
			case "satisfied":if (!this.registeredSatisfied) {this.registeredSatisfied = true;this.events.registerEvent("satisfied","callbackNodeSatisfied",this.navEngineRef,null,true);};break;
			case "score":if (!this.registeredCallbackNodeScore) {this.registeredCallbackNodeScore = true;this.events.registerEvent("score","callbackNodeScore",this.navEngineRef,null,true);};break;
			case "block":if (!this.registeredBlock) {this.registeredBlock = true;this.events.registerEvent("block","callbackNodeBlock",this.navEngineRef,null,true);};break;
			case "skip":if (!this.registeredSkip) {this.registeredSkip = true;this.events.registerEvent("skip","callbackNodeSkip",this.navEngineRef,null,true);};break;
			case "enabled":if (!this.registeredEnabled) {this.registeredEnabled = true;this.events.registerEvent("enabled","callbackNodeEnabled",this.navEngineRef,null,true);};break;
			case "visible":if (!this.registeredVisible) {this.registeredVisible = true;this.events.registerEvent("visible","callbackNodeVisible",this.navEngineRef,null,true);};break;
			case "choice":if (!this.registeredChoice) {this.registeredChoice = true;this.events.registerEvent("choice","callbackNodeChoice",this.navEngineRef,null,true);};break;
			case "flow":if (!this.registeredFlow) {this.registeredFlow = true;this.events.registerEvent("flow","callbackNodeFlow",this.navEngineRef,null,true);};break;
			case "forwardOnly":if (!this.registeredForwardOnly) {this.registeredForwardOnly = true;this.events.registerEvent("forwardOnly","callbackNodeForwardOnly",this.navEngineRef,null,true);};break;
		}
	}
	
	return true;
}